/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_get_info.c,v 1.52 2006/09/12 21:28:38 bgoglin Exp $";
 
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__error.h"
#include "mx__lib.h"
#include "mx__driver_interface.h"
#include "mx__fops.h"
#include "mx_byteswap.h"

static mx_return_t mx__nic_count(uint32_t *val, uint32_t length);
static mx_return_t mx__nic_ids(uint64_t *val, uint32_t length);
static mx_return_t mx__max_native_endpoints(uint32_t *val, uint32_t length);
static mx_return_t mx__counters_count(uint32_t *in_val, uint32_t in_len,
				      uint32_t *out_val, uint32_t out_len);
static mx_return_t mx__counters_labels(uint32_t *in_val, uint32_t in_len,
				       uint8_t out_val[][MX_MAX_STR_LEN],
				       uint32_t out_len);
static mx_return_t mx__counters_values(uint32_t *in_val, uint32_t in_len,
				       uint32_t *out_val, uint32_t out_len);
static mx_return_t mx__product_code(uint32_t *in_val, uint32_t in_len,
				    uint8_t out_val[MX_MAX_STR_LEN],
				    uint32_t out_len);
static mx_return_t mx__part_number(uint32_t *in_val, uint32_t in_len,
				   uint8_t out_val[MX_MAX_STR_LEN],
				   uint32_t out_len);
static mx_return_t mx__serial_number(uint32_t *in_val, uint32_t in_len,
				     uint32_t *out_val, uint32_t out_len);
static mx_return_t mx__port_count(uint32_t *in_val, uint32_t in_len,
				  uint32_t *out_val, uint32_t out_len);

/* TODO: should make this depend on endpoint */
static mx_return_t mx__native_requests(uint32_t *val, uint32_t length);

MX_FUNC(mx_return_t)
mx_get_info(mx_endpoint_t endpoint, mx_get_info_key_t key, void *in_val,
	    uint32_t in_len, void *out_val, uint32_t out_len)
{
  if (out_val == NULL) {
    return mx__error_noep("mx_get_info(%d)", MX_BAD_INFO_VAL, key);
  }

  /* TODO: The endpoint specific queries need a lock. */

  switch(key) {

  /* keys that don't need an endpoint */
  case MX_NIC_COUNT:
    return mx__nic_count(out_val, out_len);
  case MX_NIC_IDS:
    return mx__nic_ids(out_val, out_len);
  case MX_MAX_NATIVE_ENDPOINTS:
    return mx__max_native_endpoints(out_val, out_len);
  case MX_NATIVE_REQUESTS:
    return mx__native_requests(out_val, out_len);
  case MX_COUNTERS_COUNT:
    return mx__counters_count(in_val, in_len, out_val, out_len);
  case MX_COUNTERS_LABELS:
    return mx__counters_labels(in_val, in_len, out_val, out_len);
  case MX_COUNTERS_VALUES:
    return mx__counters_values(in_val, in_len, out_val, out_len);
  case MX_PRODUCT_CODE:
    return mx__product_code(in_val, in_len, out_val, out_len);
  case MX_PART_NUMBER:
    return mx__part_number(in_val, in_len, out_val, out_len);
  case MX_SERIAL_NUMBER:
    return mx__serial_number(in_val, in_len, out_val, out_len);
  case MX_PORT_COUNT:
    return mx__port_count(in_val, in_len, out_val, out_len);

  /* keys that don't need an endpoint */
  /* FIXME: need locking? */
  case MX_PIO_SEND_MAX:
    if (out_len < sizeof(uint32_t))
      return MX_BAD_INFO_LENGTH;
    *(uint32_t*)out_val = endpoint->small_msg_threshold;
    return MX_SUCCESS;
  case MX_COPY_SEND_MAX:
    if (out_len < sizeof(uint32_t))
      return MX_BAD_INFO_LENGTH;
    *(uint32_t*)out_val = endpoint->medium_msg_threshold;
    return MX_SUCCESS;    

  default:
    return mx__error_noep("mx_get_info(key=%d)", MX_BAD_INFO_KEY, key);
  }
}

MX_FUNC(mx_return_t)
mx_board_number_to_nic_id(uint32_t board_number, uint64_t *nic_id)
{
  mx_get_nic_id_t tmp;
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t count;

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  if (mx__get_instance_count(handle, &count) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  if (board_number >= count) {
    r = MX_BOARD_UNKNOWN;
    goto abort_with_handle;
  }

  tmp.board_number = board_number;
  if (mx__get_nic_id(handle, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }
  *nic_id = tmp.nic_id;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("mx_board_number_to_nic_id(%d)", r, board_number);
}


static mx_return_t
mx__nic_count(uint32_t *val, uint32_t length)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t i;

  if (length < sizeof(*val)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  if (mx__get_instance_count(handle, &i) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  *val = i;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/nic_count", r);
}

MX_FUNC(mx_return_t)
mx_nic_id_to_board_number(uint64_t nic_id, uint32_t *board_number)
{
  mx_return_t rc;
  mx_nic_id_to_board_num_t tmp;
  mx_endpt_handle_t handle;

  tmp.nic_id = nic_id;

  rc = mx_open_any_board(&handle);
  if (rc != MX_SUCCESS) {
    return mx__error_noep("mx_nic_id_to_board_number()", 
			  MX_BAD_BAD_BAD);
  }
  rc = mx__nic_id_to_board_num(handle, &tmp);
  mx__close(handle);
  if (rc != 0) {
    return mx__error_noep("mx_nic_id_to_board_number(%012"PRIx64")", 
			  MX_BOARD_UNKNOWN, nic_id);
  }
  *board_number = tmp.board_number;
  return MX_SUCCESS;
}

static mx_return_t mx__nic_ids(uint64_t *val, uint32_t length)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t i, count = -1;
  mx_get_nic_id_t tmp;

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  if (mx__get_instance_count(handle, &count) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  if (length < sizeof (*val) * (count+1)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_handle;
  }

  for (i = 0; i < count; ++i) {
    tmp.board_number = i;
    if (mx__get_nic_id(handle, &tmp) != 0) {
      r = MX_BAD_BAD_BAD;
      goto abort_with_handle;
    }
    val[i] = tmp.nic_id;
  }
  val[count] = 0;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/nic_ids, out_blen=%d,nic_count=%d", r, length, count);
}

static mx_return_t
mx__max_native_endpoints(uint32_t *val, uint32_t length)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t i;

  if (length < sizeof(*val)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  if (mx__get_max_endpoints(handle, &i) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  *val = i;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/native_endpoints", r);
}

static mx_return_t mx__native_requests(uint32_t *val, uint32_t length)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t i;

  if (length < sizeof(*val)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  if (mx__get_num_handles(handle, &i) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  *val = i;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/native_requests", r);
}

static mx_return_t mx__counters_count(uint32_t *in_val, uint32_t in_len,
				      uint32_t *out_val, uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  mx_get_counters_strings_t tmp;

  if ((in_len < sizeof (*in_val)) ||
      (out_len < sizeof (*out_val))) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  tmp.count = 0;
  if (mx__get_counters_strings(handle, *in_val, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

  *out_val = tmp.count;

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/counters_count", r);
}

static mx_return_t mx__counters_labels(uint32_t *in_val, uint32_t in_len,
				       uint8_t out_val[][MX_MAX_STR_LEN],
				       uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t count = 0; /* -Wunused on broken gcc-4 */
  mx_get_counters_strings_t *names;
  uint32_t i;

  if (in_len < sizeof (*in_val)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx__counters_count(in_val, in_len, &count, sizeof (count))) != MX_SUCCESS) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_nothing;
  }

  if (out_len < count * MX_MAX_STR_LEN) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  names = mx_malloc (offsetof(mx_get_counters_strings_t, label) 
		  + MX_MAX_STR_LEN * count);
  if (names == NULL) {
    r = MX_NO_RESOURCES;
    goto abort_with_handle;
  }
  names->count = count;
  if (mx__get_counters_strings(handle, *in_val, names) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_names;
  }
  for (i = 0; i < count; ++i) {
    mx_memcpy(out_val[i], names->label[i], MX_MAX_STR_LEN);
  }

 abort_with_names:
  mx_free(names);
 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/counters_labels", r);
}

static mx_return_t mx__counters_values(uint32_t *in_val, uint32_t in_len,
				       uint32_t *out_val, uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t count  = 0; /* -Wunused on broken gcc-4 */
  uint32_t counters[1024];
  uint32_t i;

  if (in_len < sizeof (*in_val)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx__counters_count(in_val, in_len, &count, sizeof (count))) != MX_SUCCESS) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_nothing;
  }

  if (out_len < count * sizeof (uint32_t)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  do {
    r = mx__get_counters(handle, *in_val, counters);
    if (r && (mx_errno != EBUSY)) {
      goto abort_with_handle;
    }
  } while (r);
  for (i = 0; i < count; ++i) {
    out_val[i] = ntohl(counters[i]);
  }

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/counters_values", r);
}

static mx_return_t mx__product_code(uint32_t *in_val, uint32_t in_len,
				    uint8_t out_val[MX_MAX_STR_LEN],
				    uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  mx_get_eeprom_string_t tmp;

  if ((in_len < sizeof (*in_val)) ||
      (out_len < MX_MAX_STR_LEN)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  tmp.board_number = *in_val;
  tmp.buffer = (uint64_t)(uintptr_t)out_val;
  if (mx__get_product_code(handle, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/product_code", r);
}

static mx_return_t mx__part_number(uint32_t *in_val, uint32_t in_len,
				   uint8_t out_val[MX_MAX_STR_LEN],
				   uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  mx_get_eeprom_string_t tmp;

  if ((in_len < sizeof (*in_val)) ||
      (out_len < MX_MAX_STR_LEN)) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  tmp.board_number = *in_val;
  tmp.buffer = (uint64_t)(uintptr_t)out_val;
  if (mx__get_part_number(handle, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }

 abort_with_handle:
  mx__close(handle);
 abort_with_nothing:
  return mx__error_noep("get_info/part_number", r);
}

static mx_return_t mx__serial_number(uint32_t *in_val, uint32_t in_len,
				     uint32_t *out_val, uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t tmp;

  if ((in_len < sizeof (*in_val)) ||
      (out_len < sizeof (*out_val))) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  tmp = *in_val;
  if (mx__get_serial_number(handle, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }
  *out_val = tmp;

  abort_with_handle:
    mx__close(handle);
 abort_with_nothing:
  return r;
}

static mx_return_t mx__port_count(uint32_t *in_val, uint32_t in_len,
				  uint32_t *out_val, uint32_t out_len)
{
  mx_return_t r = MX_SUCCESS;
  mx_endpt_handle_t handle;
  uint32_t tmp;

  if ((in_len < sizeof (*in_val)) ||
      (out_len < sizeof (*out_val))) {
    r = MX_BAD_INFO_LENGTH;
    goto abort_with_nothing;
  }

  if ((r = mx_open_any_board(&handle)) != MX_SUCCESS) {
    goto abort_with_nothing;
  }

  tmp = *in_val;
  if (mx__get_num_ports(handle, &tmp) != 0) {
    r = MX_BAD_BAD_BAD;
    goto abort_with_handle;
  }
  *out_val = tmp;

  abort_with_handle:
    mx__close(handle);
 abort_with_nothing:
  return r;
}

#ifdef MX_VERSIONED_SYMS
/* binary compatibility with previous version */

mx_return_t
mx_get_info_v1(mx_endpoint_t endpoint, mx_get_info_key_t key, void *out_val, uint32_t out_len)
{
  return mx_get_info(endpoint, key, NULL, 0, out_val, out_len);
}
__asm__(".symver mx_get_info_v1,mx_get_info@MX_0.0");

#endif
